#include <stdio.h>      /* for fopen and friends */
#include <ctype.h>
#include "string.h"
#include "hdr/global.h"
#include "hdr/routines.h"




void ChangeRuleset(int ruleset)
{
	if (ruleset>0 && ruleset<10) {
		Ruleset = ruleset;
		pushWin3("Active ruleset changed to ruleset %d.", Ruleset);
	} else if (ruleset == -1) {
		pushWin3("Active ruleset is ruleset %d.", Ruleset);
	}
}




void ClearRuleset(int ruleset)
{
	register int i, rule;

	if (ruleset) {
		for (i=0; i<26; ++i) {
			Letter[i].rule[ruleset] = ' ';
			Letter[i].mask[ruleset] = ' ';
		}
		pushWin3("Cleared ruleset %d.", ruleset);
	} else {
		/* Clear all rulesets (ruleset = 0) */
		for (i=0; i<26; ++i) {
			for (rule=0; rule<10; ++rule) {
				Letter[i].rule[rule] = ' ';
				Letter[i].mask[rule] = ' ';
			}
		}
		pushWin3("Cleared all rulesets.");
	}
}




void SaveState(char *a)
{
	FILE *fp;
	GString *filename;
	register int i;

	filename = g_string_new("");
	g_string_assign(filename, a);

	if (strstr(filename->str, ".lcs") == NULL)
		g_string_append(filename, ".lcs");

	if ((fp=fopen(filename->str, "w+b"))==NULL)
		pushWin3("Can't open file %s.", filename->str);
	fwrite(&ctl, sizeof(control), 1, fp);
	for(i=0; i<26; i++)
		fwrite(&Letter[i], sizeof(LetterStruct), 1, fp);

	if ((i=fclose(fp)) != 0)
		pushWin3("Error %d in closing file %s.", i, filename->str);
	else
		pushWin3("Successfully saved state to %s.", filename->str);

	g_string_free(filename, TRUE);
}




void RestoreState(char *a) {
	FILE *fp;
	register int i;
	GString *filename;

	RestoredState = TRUE;
	filename = g_string_new("");
	g_string_assign(filename, a);
	if (strstr(filename->str, ".lcs") == NULL)
			g_string_append(filename, ".lcs");

	if ((fp = fopen(filename->str, "r+b"))==NULL)
		die(1, "Can't open file to restore state.");

	fread(&ctl, sizeof(control), 1, fp);
	for(i=0; i<26; i++)
		fread(&Letter[i], sizeof(LetterStruct), 1, fp);

	/* when we restore state, it shows the last command before saving state,
      which is confusing.  just better to null out window 3 */
	for (i=1; i<W3LEN; i++)
		g_string_assign(Win3[i], "");

	if ((i=fclose(fp)) != 0)
		pushWin3("Error %d in closing file %s.", i, filename->str);
	else
		pushWin3("Successfully restored state from %s.", filename->str);

	g_string_free(filename, TRUE);
}




void Assign(char from, char to, const int ruleset) {
	from = toupper(from) - 'A';
	to   = toupper(to);

	if (ruleset == 0) {
		register int i;
		for(i=1; i<10; i++) Letter[(int)from].rule[i] = to;
		pushWin3("Mapped %c->%c in all rulesets.", from + 'A', to);
	} else if (ruleset == -1) {
		Letter[(int)from].rule[Ruleset] = to;
		pushWin3("Mapped %c->%c in the current ruleset.", from + 'A', to);
	} else {
		Letter[(int)from].rule[ruleset] = to;
		pushWin3("Mapped %c->%c in ruleset %d.", from + 'A', to, ruleset);
	}
}




void ParseError(char *err)
{
	pushWin3("I don't know what %s means.", err);
}





/*   from can be 1-9.  to can be 0-9 (0 means all rulesets)
 */
void CopyRuleset(int from, int to) {
	int i, j;

	if (from == -1) from = Ruleset;

	if (from < 1 || from > 9)
		pushWin3("Badly formed source ruleset");
	else if (to < 0 || to >9 )
		pushWin3("Badly formed destination ruleset");

	if ( to != 0 ) {
		for(i=0; i<26; i++) {
			Letter[i].rule[to] = Letter[i].rule[from];
			Letter[i].mask[to] = Letter[i].mask[from];
		}
		pushWin3("Ruleset %d copied to %d.", from, to);
	} else {
		for(i=0; i<26; i++) {
			for(j=1; j<10; j++) {
				Letter[i].rule[j] = Letter[i].rule[from];
				Letter[i].mask[j] = Letter[i].mask[from];
			}
		}
		pushWin3("Copied ruleset %d to all rulesets.", from);
	}
}




void WordAssign(const char *from, const char *to, int ruleset)
{
	register int i, j;
	GString *f, *t;

	f = g_string_new(from);
	t = g_string_new(to);

	f = g_string_up(f);
	t = g_string_up(t);

	if (ruleset == -1) ruleset = Ruleset;

	/* Check for non balanced wrds and non alphas */
	if (f->len != t->len) {
		pushWin3("Error: <%s> and <%s> aren't balanced.", f->str, t->str);
		return;
	}

	/* Check to see if there are non-alpha characters in the words */
	for(i=0; i<f->len; i++) {
		if ( ! isalpha(*(from+i)) ) {
			pushWin3("Error: %s contains non-alpha characters", f->str); return;
		} else if ( ! isalpha(*(to+i)) ) {
			pushWin3("Error: %s contains non-alpha characters", t->str); return;
		}
	}

	for(i=0; i<f->len; i++) {
		if (! ruleset)
			for(j=1; j<10; j++) Letter[t->str[i]-'A'].rule[j] = f->str[i];
		else
			Letter[t->str[i]-'A'].rule[ruleset] = f->str[i];
	}

	switch(ruleset) {
		case 0:
			pushWin3("Assigned %s to %s in all rulesets.", f->str, t->str);
			break;
		default:
			pushWin3("Assigned %s to %s in ruleset %d.", f->str, t->str, ruleset);
			break;
	}
}	



#include <ncurses.h>
void PrintScreen(void) {
	FILE *pipe; 
	int row, col, cur_x, cur_y, cur_state, err=0;


	getsyx(cur_y, cur_x);
	cur_state = curs_set(0);
	fflush(stdout);

	pipe = popen("lpr", "w");
	if (pipe == NULL) {
		err=1;
	} else {
		for (row=0; row < LINES; ++row) {
			for (col=0; col<COLS; ++col) {
				/* fputc(A_CHARTEXT & mvinch(row,col), pipe); */
				fputc(0xff & mvinch(row,col), pipe);
			}
			fputc('\n', pipe);
		}
		if(pclose(pipe)==-1) err=2;
		setsyx(cur_y, cur_x);
		curs_set(cur_state);
		refresh();
	}
	switch(err) {
		case 0:  pushWin3("Screen printed."); break;
		case 1:  pushWin3("popen() failed."); break;
		case 2:  pushWin3("pclose() failed."); break;
		default: pushWin3("I shouldn't be here!"); break;
	}

	mcprint("hello", strlen("hello")*sizeof(char));
}



void Mark(const char a, const char *marktype, int ruleset)
{
	register int j, index = toupper(a) - 'A';
	char mask;

	if (ruleset == -1) ruleset = Ruleset;

	if      (! strcmp(marktype, "vowel"))     { mask = VOWEL; }
	else if (! strcmp(marktype, "consonant")) { mask = CONSONANT; }
	else if (! strcmp(marktype, "exact"))     { mask = EXACT; }
	else if (! strcmp(marktype, "inexact"))   { mask = INEXACT; }

	if(ruleset == 0) {
		for(j=1; j<10; j++) Letter[index].mask[j] = mask;
		pushWin3("Marked %c as %s in all rulesets.", toupper(a), marktype);
	} else {
		Letter[index].mask[ruleset] = mask;
		pushWin3("Marked %c as %s in ruleset %d.", toupper(a), marktype, ruleset);
	}
}



void UnMark(const char a, int ruleset)
{
	register int j, index = toupper(a) - 'A';

	if(ruleset == 0) {
		for(j=1; j<10; j++) Letter[index].mask[j] = ' ';
		pushWin3("Unmarked %c in all rulesets.", toupper(a));
	} else {
		Letter[index].mask[ruleset] = ' ';
		pushWin3("Marked %c in ruleset %d.", toupper(a), ruleset);
	}
}



void Win2Sort(const char *type)
{
	if (! strcmp(type, "ctxt")) {
		Sort = BY_CIPHERTEXT;
		pushWin3("Sorting by ciphertext");
	} else if (! strcmp(type, "freq")) {
		Sort = BY_FREQUENCY;
		pushWin3("Sorting by frequency");
	} else if (! strcmp(type, "ptxt")) {
		Sort = BY_PLAINTEXT;
		pushWin3("Sorting by plaintext");
	} else {
		die(1, "oops from Win2Sort");
	}
}



void ChangeWin2Mode(const char *type)
{
	if (! strcmp(type, "all")) {
		Win2Mode = SHOW_FREQ;
		pushWin3("Showing frequency of all letters");
	} else if (! strcmp(type, "initial")) {
		Win2Mode = SHOW_INITIAL_FREQ;
		pushWin3("Showing frequency of initial letters");
	} else if (! strcmp(type, "final")) {
		Win2Mode = SHOW_ENDING_FREQ;
		pushWin3("Showing frequency of final letters");
	}
}



void SetDictionary(const char *dictionary)
{
	g_string_assign(ctl.dictionary, dictionary);
	pushWin3("Changed dictionary to %s", ctl.dictionary->str);
}



void FindAlphaParse(const char *word)
{
	GString *Word, *WordAlpha, *Target, *TargetAlpha;
	FILE *fp;
	GSList *TargetList = NULL;


	/*  If I can find out how to read a file into a GString, this would not
	 *  be necessary!
	 */
	gchar string[INPUT_SZ];

	Word        = g_string_new(word);
	WordAlpha   = g_string_new("");
	Target      = g_string_new("");
	TargetAlpha = g_string_new("");

	AlphaParse(Word, WordAlpha);
	pushWin3("alpha parse for %s is %s.", Word->str, WordAlpha->str);

	if ((fp = fopen(ctl.dictionary->str, "r")) == NULL)
		die(1, "fopen error in FindAlphaParse");

	while (fgets(string, INPUT_SZ - 1, fp) != NULL) {
		/*  fgets reads till \n or EOF but stores \n.  Convert them to \0 */
		g_strdelimit(string, "\n", '\0');
		/* Convert string to a GString and get its alpha parse. */
		g_string_assign(Target, string);
		Target = g_string_up(Target);
		AlphaParse(Target, TargetAlpha);

		/* Case insensitive string comparison */
		if (! g_strcasecmp(WordAlpha->str, TargetAlpha->str)) {
			gchar *element = g_strdup(Target->str);
			TargetList = g_slist_append(TargetList, element);
		}

	}
	fclose(fp);
	g_slist_free(TargetList);
}
